<?php

namespace Vendor\CustomSearch\Console;

use Magento\Framework\App\State;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\File\Csv;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\Product\Attribute\Source\Status;

class InsertProductData extends Command
{
    protected $productRepository;
    protected $connection;
    protected $directoryList;
    protected $csvProcessor;
    protected $productCollectionFactory;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        ResourceConnection $resourceConnection,
        DirectoryList $directoryList,
        Csv $csvProcessor,
        CollectionFactory $productCollectionFactory
    ) {
        $this->productRepository = $productRepository;
        $this->connection = $resourceConnection->getConnection();
        $this->directoryList = $directoryList;
        $this->csvProcessor = $csvProcessor;
        $this->productCollectionFactory = $productCollectionFactory;
        parent::__construct();
    }

    protected function configure()
    {
        $this->setName('sematic:dataexport:export')
            ->setDescription('Export Catalog Product Data to CSV');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            // Fetch all product data
            $productCollection = $this->productCollectionFactory->create();
            
            // Standard approach for filtering by status and visibility using constants
            $productCollection->addFieldToFilter('status', ['eq' => Status::STATUS_ENABLED]); // Filter enabled products
            $productCollection->addFieldToFilter('visibility', ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_CATALOG]]); // Visible in Catalog or Search
            
            // Include attributes in the collection
            $productCollection->addAttributeToSelect(['name', 'short_description', 'sku', 'gpn', 'mgs_brand', 'price', 'lead_time']);
            
            // Prepare CSV data
            $data = [];
            // $header = ['product_id', 'product_name', 'description', 'sku', 'gpn', 'mfc', 'price', 'lead_time'];
            $header = ['product_id', 'product_name', 'description', 'sku', 'gpn', 'mfc', 'lead_time'];
            $data[] = $header; // Add header to CSV

            foreach ($productCollection as $product) {
                // Prepare product data
                $textWithoutHtml = strip_tags($product->getShortDescription());
                $cleanedText = str_replace("'", "", $textWithoutHtml);

                $data[] = [
                    'product_id' => $product->getId(),
                    'product_name' => $product->getName(),
                    'description' => $cleanedText,
                    'sku' => $product->getSku(),
                    'gpn' => $product->getGpn(),
                    'mfc' => $product->getAttributeText('mgs_brand'),
                    // 'price' => $product->getPrice(),
                    'lead_time' => $product->getLeadTime() ? $product->getLeadTime() : ''
                ];

                $output->writeln("Exporting Product : " . $product->getName()."\n");
            }

            // Define file path to save the CSV file
            $fileName = 'products_export_' . date('Y-m-d_H-i-s') . '.csv';
            $filePath = $this->directoryList->getPath('var') . '/export/' . $fileName;
            // echo $filePath; die;
            // Ensure the export directory exists
            $exportDir = dirname($filePath);
            if (!is_dir($exportDir)) {
                mkdir($exportDir, 0777, true);
            }

            // Write data to the CSV file
            $this->csvProcessor->saveData($filePath, $data);

            // Output success message
            $output->writeln("Count: ".$productCollection->getSize()." -- Product data has been successfully exported to CSV: " . $filePath);

            return \Magento\Framework\Console\Cli::RETURN_SUCCESS;

        } catch (\Exception $e) {
            $output->writeln("Error: " . $e->getMessage());
            return \Magento\Framework\Console\Cli::RETURN_FAILURE;
        }
    }
}