<?php

/**
 * @author Paras Suryawanshi
 */

namespace Vendor\CustomSearch\Cron;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Psr\Log\LoggerInterface;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\Product\Attribute\Source\Status;

class ExportData
{
    protected $productRepository;
    protected $connection;
    protected $productCollectionFactory;
    protected $logger;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        ResourceConnection $resourceConnection,
        CollectionFactory $productCollectionFactory,
        LoggerInterface $logger
    ) {
        $this->productRepository = $productRepository;
        $this->connection = $resourceConnection->getConnection();
        $this->productCollectionFactory = $productCollectionFactory;
        $this->logger = $logger;
    }

    public function execute()
    {
        try {
            // logs in system.log
            $this->logger->info('Semantic Custom cron job executed at: ' . date('Y-m-d H:i:s'));

            // Fetch product collection
            $productCollection = $this->productCollectionFactory->create();

            // Filter enabled products & those visible in catalog/search
            $productCollection->addFieldToFilter('status', ['eq' => Status::STATUS_ENABLED]);
            $productCollection->addFieldToFilter('visibility', ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_CATALOG]]);

            // Select necessary attributes
            $productCollection->addAttributeToSelect(['name', 'short_description', 'sku', 'gpn', 'mgs_brand', 'lead_time']);

            foreach ($productCollection as $product) {
                // Clean description text
                $cleanedText = strip_tags($product->getShortDescription());
                $cleanedText = str_replace("'", "", $cleanedText);

                // Prepare data array
                $productData = [
                    'product_id' => $product->getId(),
                    'product_name' => $product->getName(),
                    'description' => $cleanedText,
                    'sku' => $product->getSku(),
                    'gpn' => $product->getGpn(),
                    'mfc' => $product->getAttributeText('mgs_brand'),
                    'lead_time' => $product->getLeadTime() ?? ''
                ];

                // Check if product exists in sematic_data table
                $query = "SELECT COUNT(*) FROM sematic_data WHERE product_id = :product_id";
                $count = $this->connection->fetchOne($query, ['product_id' => $productData['product_id']]);

                if ($count > 0) {
                    // Update existing record
                    $updateQuery = "UPDATE sematic_data SET 
                                        product_name = :product_name,
                                        description = :description,
                                        sku = :sku,
                                        gpn = :gpn,
                                        mfc = :mfc,
                                        lead_time = :lead_time
                                    WHERE product_id = :product_id";
                    $this->connection->query($updateQuery, $productData);
                    $this->logger->info("Semantic Updated Product: " . $product->getGpn());
                } else {
                    // Insert new record
                    $insertQuery = "INSERT INTO sematic_data (product_id, product_name, description, sku, gpn, mfc, lead_time) 
                                    VALUES (:product_id, :product_name, :description, :sku, :gpn, :mfc, :lead_time)";
                    $this->connection->query($insertQuery, $productData);
                    $this->logger->info("Semantic Inserted Product: " . $product->getGpn());
                }
            }

            $this->logger->info("Semantic Total Products Processed: " . $productCollection->getSize());
        } catch (\Exception $e) {
            $this->logger->error("Semantic Cron Job Error: " . $e->getMessage());
        }
    }
}
